<?php

/*
 *******************
 * INCLUDE ACTIVATION / TRIAL ASSETS
 *******************
 *
 *	Function to enqueue important assets for the trial period.
 *
 *  @type	action
 *  @date	06/18/19
 *  @since	2.0
 *
 *  @param	N/A
 *  @return	N/A
 *
*/

if ( ! function_exists( 'wphave_admin_trial_activation_js' ) ) :

	function wphave_admin_trial_activation_js( $hook ) {

		// Method to get the page hook
		//wp_die($hook);

		// Load the following JS files only on theme welcome page
		if( $hook != 'tools_page_wphave-admin-purchase-code' ) {
			return;
		}

		wp_enqueue_script( 
			'wphave-admin-activation-trial', wphave_admin_path( 'inc/init/setup/assets/js/setup_trial.js' ), array( 'jquery' ), null, true 
		);

		wp_localize_script( 
			'wphave-admin-activation-trial', 'wp_ajax_data', array( 'wp_ajax_url' => admin_url( 'admin-ajax.php' ),
		));

		// Localize the script with new data
		$vars = wphave_admin_activation_localized_script_vars();
		wp_localize_script( 'wphave-admin-activation-trial', 'WP_JS_Theme_Activation', $vars );

	}

endif;

add_action( 'admin_enqueue_scripts', 'wphave_admin_trial_activation_js', 30 );


/*
 *******************
 * CHECK THEME TRIAL ATTEMPTS
 *******************
 *
 *	Function to check the theme trial period attempts.
 *
 *  @type	action
 *  @date	06/18/19
 *  @since	2.0
 *
 *  @param	N/A
 *  @return	N/A
 *
*/

if ( ! function_exists( 'wphave_admin_trial_attempts' ) ) :

	function wphave_admin_trial_attempts() {

		$option = get_option('wphave_data');
		
		// Add trial attempts option to saved trail attempts count
		$attempt_count = isset( $option['wphave_admin_trial_attempts'] ) ? $option['wphave_admin_trial_attempts'] : '';	

		// Check if trial attempts option exist
		if( $attempt_count != '' ) {
			// Count up by 1 after each trial attempt
			$option['wphave_admin_trial_attempts'] = $attempt_count+1;
		} else {
			// Start from 1
			$option['wphave_admin_trial_attempts'] = '1';
		}

		// Update trial attempts count
		update_option( 'wphave_data', $option, 'yes' );

		return $attempt_count;

	}

endif;


/*
 *******************
 * CHECK THEME TRIAL PERIOD EXPIRATION
 *******************
 *
 *	Function to check the theme trial period expiration.
 *
 *  @type	action
 *  @date	06/18/19
 *  @since	2.0
 *
 *  @param	N/A
 *  @return	N/A
 *
*/

if ( ! function_exists( 'wphave_admin_trial_period' ) ) :

	function wphave_admin_trial_period( $attempt_count ) {

		$option = get_option( 'wphave_data' );
		
		// Add trial period option to save trial period		
		$trial_period = isset( $option['wphave_admin_trial_period'] ) ? $option['wphave_admin_trial_period'] : '';

		// Get current date/time
		$current_datetime = current_time( 'mysql' );
		
		// Update trial period option (if trial period is expired)
		if( $trial_period == '' || $attempt_count <= 5 ) {

			$data = array();		
			$data['trial_start_time'] = $current_datetime;
			// Define trial period (14400 seconds = 4 hours)
			$data['trial_end_time'] = date( 'Y-m-d H:i:s', strtotime( $current_datetime ) + 14400 );

			// Update trial period
			$option['wphave_admin_trial_period'] = $data;

			// Enable trial period and set to active
			$option['wphave_admin_trial_status'] = 'active';

			// Enable temporary admin license and set to active	
			$option['wphave_admin_license'] = 'active';
			
			// Update option
			update_option( 'wphave_data', $option, 'yes' );

			return $data;

		}

		return false;

	}

endif;


/*
 *******************
 * THEME TRIAL ACTIVATION STATUS
 *******************
 *
 *	Function to check if the trial period is currently active or expired.
 *
 *  @type	action
 *  @date	06/18/19
 *  @since	2.0
 *
 *  @param	N/A
 *  @return	N/A
 *
*/

if ( ! function_exists( 'wphave_admin_trial_status' ) ) :

	function wphave_admin_trial_status() {

		$trial_status = get_option( 'wphave_data' );
		$trial_status = isset( $trial_status['wphave_admin_trial_status'] ) ? $trial_status['wphave_admin_trial_status'] : '';

		if( $trial_status === 'active' ) {
			// Trial period is active
			return true;
		}

		// Trial period is expired
		return false;

	}

endif;


/*
 *******************
 * BREAK ACTIVE THEME TRIAL PERIOD
 *******************
 *
 *	Function to break the active trial period if the time to trial is expired.
 *
 *  @type	action
 *  @date	06/18/19
 *  @since	2.0
 *
 *  @param	N/A
 *  @return	N/A
 *
*/

if ( ! function_exists( 'wphave_admin_trial_period_check' ) ) :

	function wphave_admin_trial_period_check() {	

		$option = get_option( 'wphave_data' );
		
		// Get trial period data
		$trial_period = isset( $option['wphave_admin_trial_period']['trial_end_time'] ) ? $option['wphave_admin_trial_period']['trial_end_time'] : '';	

		// Get current date/time
		$current_datetime = current_time( 'mysql' );	

		// Check for active trial period
		if( wphave_admin_trial_status() && $current_datetime > $trial_period ) {
			
			// Disable trial status and set to inactive
			$option['wphave_admin_trial_status'] = 'inactive';

			// Disable temporary admin license and set to inactive
			$option['wphave_admin_license'] = 'inactive';
			
			// Update option
			update_option( 'wphave_data', $option, 'yes' );
			
		}

		return false;

	}

endif;

add_action('admin_init', 'wphave_admin_trial_period_check');


/*
 *******************
 * THEME TRIAL PERIOD - CALCULATE TIME LEFT
 *******************
 *
 *	Function to calculate the time left until the trial period is expired.
 *
 *  @type	action
 *  @date	06/18/19
 *  @since	2.0
 *
 *  @param	N/A
 *  @return	N/A
 *
*/

if ( ! function_exists( 'wphave_admin_trial_calc_time_left' ) ) :

	function wphave_admin_trial_calc_time_left() { 

		$option = get_option('wphave_data');
		
		// Get time difference until the trial period is expired
		$trial_period = isset( $option['wphave_admin_trial_period'] ) ? $option['wphave_admin_trial_period'] : '';

		$time_left = 'n/a';

		if( $trial_period != '' ) {
			$start_time = strtotime( current_time( 'mysql' ) );
			$end_time = strtotime( $trial_period['trial_end_time'] );
			$time_left = round( ( $end_time - $start_time ) / 60 );		
		}	

		return $time_left;

	}

endif;


/*
 *******************
 * THEME TRIAL PERIOD ADMIN NOTICE
 *******************
 *
 *	Function to add an admin notice about the current trial period.
 *
 *  @type	action
 *  @date	06/18/19
 *  @since	2.0
 *
 *  @param	N/A
 *  @return	N/A
 *
*/

if ( ! function_exists( 'wphave_admin_trial_admin_notice' ) ) :

	function wphave_admin_trial_admin_notice() { 

		if( ! wphave_admin_trial_status() ) {
			return false;
		}

		// Get time difference until the trial period is expired
		$time_left = wphave_admin_trial_calc_time_left();

		$plugin_activate_link = '<a href="' . admin_url('tools.php?page=wphave-admin-purchase-code&tab=activation') . '">' . __( 'Enter your purchase code here', 'wphave-admin' ) . '</a>.'; ?>

		<div class="notice notice-warning">
			<p><?php printf( wp_kses_post( __( 'You are using the trial version of this theme. There are <strong>still %2$s minutes left</strong> until the end of the test period. %1$s', 'wphave-admin' ) ), $plugin_activate_link, $time_left ); ?></p>
		</div>

	<?php }

endif;

add_action('admin_notices', 'wphave_admin_trial_admin_notice');


/*
 *******************
 * UNLOCK THEME FOR TRIAL PERIOD AJAX
 *******************
 *
 *	Function to unlock the theme trial period via AJAX.
 *
 *  @type	action
 *  @date	06/18/19
 *  @since	2.0
 *
 *  @param	N/A
 *  @return	N/A
 *
*/

if ( ! function_exists( 'wphave_admin_trial_process' ) ) :

	function wphave_admin_trial_process() {

		// --> !!! For security check and verify wp_nonce() before saving new values from AJAX request
		check_ajax_referer( 'verify_sec_request', 'security' );

		if( $_POST ) {		

			// Get client action to start trial period
			$request = json_decode( stripslashes( $_POST['fieldData'] ) );

			if( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
				$action = $request[0]->command;
			}

			$trial_notice = __( 'Something went wrong', 'wphavewphave-admin' );
			$trial_action = 'error';

			// 1. Check for trial unlocking license request
			if( $action == 'start_plugin_trial_period' ) {

				// 2. Count up trial attempts
				$attempt_count = wphave_admin_trial_attempts();

				// 3. Trial period was started
				$trial_notice = __( 'The test period was started', 'wphavewphave-admin' );
				$trial_action = 'success';				

				// 4. End of trial period reached
				if( $attempt_count >= 5 ) {
					$trial_notice = __( 'The maximum test attempts are exhausted', 'wphave-admin' );
					$trial_action = 'expired';			
				}

			}

			wp_send_json( array(
				//'success' => 'SUCCESS',
				'trial_attempts' => $attempt_count,
				'trial_period' => wphave_admin_trial_period( $attempt_count ),
				'trial_notice' => $trial_notice,
				'trial_action' => $trial_action,
			) );			

		}

		die();

	}

endif;

add_action( 'wp_ajax_wphave_admin_trial_process', 'wphave_admin_trial_process' );
add_action( 'wp_ajax_nopriv_wphave_admin_trial_process', 'wphave_admin_trial_process' );